classdef vct3BoundingBox < handle
    %Bounding Box for vct3 Objects
    %   
    % B = vct3BoundingBox(vL,vU) Construct from corners
    % B = vct3BoundingBox(B)     Copy constructor
    %
    % B.Add(v)              Enlarges corners of B to include vct3 v
    % B.Add(V)              Enlarges corners of B to include vct3Array V
    % B.Add(B2)             Enlarges corners of B to include corners of B2
    %
    % B = plus(B1,d)        Expands B by d in all directions
    % B = plus(B1,B2)       Returns vct3BoundingBox(B1.vMin+V2.vMin, ...
    %                                               B1.vMax+V2.vMax)
    %
    % TF = B.Includes(v)    Returns true iff v is inside B
    % TF = B.Includes(v,d)  Returns true iff v is inside B expanded by d
    % TF = B.Includes(V)    Returns true iff every v in V is inside B
    % TF = B.Includes(V,d)  Returns true iff every v in V is inside B
    %                       expanded by d
    % TF = B.Includes(B2)   Returns true iff all of B2 is inside B
    % TF = B.Overlaps(V)    Returns true iff any of V is inside B
    % TF = B.Overlaps(B2)   Returns true iff any of B2 is inside B
    %
    % V = B.Sample(N)       Uniformly sample N vct3 entries from B
    % v = B.Sample()        Return one vct3 uniformly sampled from B
    % 
    % Copyright (C) Russell H. Taylor 2013
    % For use with CIS I only
    % Do not redistribute without written permission from Russell Taylor

        properties
        vMin = vct3( inf, inf, inf) % Lower bound corner
        vMax = vct3(-inf,-inf,-inf) % Upper bound corner
    end
    
    methods
        function B = vct3BoundingBox(vL,vU)
            switch nargin 
                case 0
                    return
                case 1    
                    switch class(vL)
                        case 'vct3BoundingBox'
                            % copy constructor, vL is a BB
                            B.vMin = vL.vMin;
                            B.vMax = vL.vMax;
                        otherwise
                            B.Add(vL);
                    end
                case 2
                    B.vMin = vL;
                    B.vMax = vU;
            end
        end
        
        function B = plus(B1,B2)
            if isa(B2,'vct3BoundingBox')
                B = vct3BoundingBox(B1.vMin+B2.vMin,B1.vMax+B2.vMax);
            else
                B = vct3BoundingBox(B1.vMin-B2,B1.vMax+B2);
            end
        end
        
        function  Add(B,v)
            switch class(v)
                case 'vct3'
                    B.vMin = vct3(min(v.el,B.vMin.el));
                    B.vMax = vct3(max(v.el,B.vMax.el));
                case 'vct3Array'
                    N = size(v.el,2);
                    for i = 1:N
                        B.Add(v(i));
                    end
                case 'vct3BoundingBox'
                    B.add(v.vMin);
                    B.add(v.vMax);
                otherwise
                    error('BadArg','Bad argument to vct3BoundingBox.Add',v);
            end
        end
        
        function TF = Includes(B,v,d)
            if nargin < 3
                d = 0;
            end
            switch class(v)
                case 'vct3'
                    TF = all(v.el<=B.vMax.el+d) && all(v.el>=B.vMin.el-d);
                case 'vct3Array'
                    N = size(v.el,2);
                    for i = 1:N
                        TF = B.Includes(B.v(i));
                        if ~TF 
                            return;
                        end
                    end
                case 'vct3BoundingBox'
                    TF = B.Includes(v.vMin) && B.Includes(v.vMax);
                otherwise
                     error('BadArg','Bad argument to vct3BoundingBox.Includes',v);
            end
        end
        
        function TF = Overlaps(B,v)
            switch class(v)
                case 'vct3'
                    TF = all(v.el<=B.vMax.el) && all(v.el>=B.vMin.el);
                case 'vct3Array'
                    N = size(v.el,2);
                    for i = 1:N
                        TF = B.Includes(B.v(i));
                        if TF 
                            return;
                        end
                    end
                case 'vct3BoundingBox'
                    TF = B.Includes(v.vMin) || B.Includes(v.vMax) || ...
                         v.Includes(B.vMin) || v.Includes(B.vMax);
                otherwise
                     error('BadArg','Bad argument to vct3BoundingBox.Includes',v);
            end
        end
        
        function V = Sample(B,N)
            if nargin<2
                V = vct3.rand(B.vMin,B.vMax);
                return;
            else
                V = vct3Array.rand(N,B.vMin,B.vMax);
            end
        end
        
        function disp(B)
            disp('vMin'); disp(B.vMin.el');
            disp('vMax'); disp(B.vMax.el');
        end
        
        function display(B)          
            if isequal(get(0,'FormatSpacing'),'compact')
               disp([inputname(1) ' =']);
               disp(B);
            else
               disp(' ')
               disp([inputname(1) ' =']);
               disp(' ');
               disp(B);
            end
        end
    end
    
end

